<?php

/* TABLE Class : May 2014
 * Use tables to create objects that can be easily displayed in posts. 
 * Tables are typically output as html and implemented as dataTables with js. 
 * Tables contain DOM info - so JS knows where to put them on the page. 
 * They contain parameters to initialise the dataTables behaviour. 
 */
 
 namespace AlphaSelect;

class Table {
	
	//The members are :
	private $html; //The actual mark up for the table, which will include ids etc. This is typically made elsewhere for now.
    private $data; //Array where each row has keys that match the columns or are at least a subset of the keys.
    private $cols; //key value pairs for the column data. The keys here must pair, or be a subset of the keys in the data which is the grid.
	private $divID; //Where we put the table and messages in the DOM. <div> Table, msgs etc</div>
	private $tDOM; //The id of the table itself - used in dataTables initialisation
	private $dTableParams; //The array of parameters to use with dataTables - mean JS is more View like and this code more Model like.
	private $msg; //Any msgs to be captured or displayed in the DOM.
	private $pParams;
    private $controlType;
    private $rowParams;
    private $tooltips;


	function __construct($tableData, $divID, $tDOM, $msg){
		//This is how new Tables are created.
        $this->setTableData($tableData);
		$this->setDivWrapper($divID);
		$this->setTableID($tDOM);
		$this->setDataTableParams();//Sets the defaults
		$this->setMsg($msg);
	}
	
	/* 
	 * SETs the html for the table
	 */
	public function setTableData($tableData){
        //We expect this to be arranged as keys are the column headings, then rows are each element of the array..
        //We assume that table[0]= array (k1=>v1, etc)
        //Cols could be changed to k1=>name1
        $this->data = $tableData;
	}

    public function setCols($cols){
        $this->cols = $cols;
        //cols should be of the form k1=>name1 with k1 matching the id in this->data
    }

    public function setRowParams($cols){
        $this->rowParams = $cols;
    }

    public function setTooltips($cols){
        $this->tooltips = $cols;
    }

    private function getRowParamsForTR(){
        return (count($this->rowParams)>0) ? "mxrAS_rowParams = " . implode(" ", $this->rowParams) : "";
    }

	/*
	 * SETs the div wrapper for the table and other output
	 */
	public function setDivWrapper($divID){
		$this->divID = $divID;
	}	
	
	/* 
	 * SETs the ID of the table - for now 
	 * only the tDOM but 
	 * TODO - change the html too so <table id="newTDOM">
	 */
	public function setTableID($tDOM){
		$this->tDOM = $tDOM;
	}
	
	/*
	 * SETs the DataTable Parameters
	 */
	public function setDataTableParams($dTableParams = null){
        if(is_null($dTableParams)){
            //Set the defaults...
            $dTableParams = array('bJQueryUI'=> false,
                'bPaginate'=> false,
                'bFilter'=> false,
                'bSort'=> true,
                'sScrollY'=>"500px",
                'dom'=>"T<'clear'>lfrtip",
                'tableTools'=>array('sSwfPath'=>"/swf/copy_csv_xls_pdf.swf")
            );
        }
        $this->dTableParams = $dTableParams;
	}

    public function tableIDs($tDom, $tDiv){
        $this->tDOM = $tDom;
        $this->divID = $tDiv;
    }
	
	/*
	 * SETs the Msg that can come packaged with the table
	 */
	public function setMsg($msg){
		$this->msg = $msg;
	}
	
	/*
	 * GETs the output array to send to the post.
	 */
    public function getHeader($hidden = []){
        //Uses the columns
        //Builds a row of headers based on $tableData
        $outputHtml="<thead><tr>";

        //Handle hidden data via the hidden array.

        foreach($this->cols as $k=>$v){
            if(in_array($k,$hidden)){
                $pre="<th class = 'hidden'>";
            } else {
                $pre="<th>";
            }
            $outputHtml .= "{$pre} {$v}</th>";
        }

        return $outputHtml;
    }

    public function getFooter(){
        return "";
    }

    public function getBody(){

        $tableStr = "<tbody>";

        //We need the cols set first.
        //Otherwise there is an error.

        //We use the $tableData keys...as the row name
        $rID = 0;
        $paramsToAdd = $this->getRowParamsForTR();

        foreach($this->data as $h=>$row){
            $cID=0;
            //If the cols have a second param then we don't add it as a column - it's a param.
            $tableStr .="<tr id='R" . ++$rID ."' class='mxrAS_rowName R$rID' {$paramsToAdd}>";

            //OK now the tricky part for every id in col we pull out the data in tableData.
            foreach($this->cols as $colKey=>$colName){
                $tableStr .= "<td id='R" . $rID . "C" . ++$cID ."'" . $this->getTooltip($colKey,$h) . ">" . $row[$colKey] . "</td>";
            }
            $tableStr .= "</tr>";
            $rID++;
        }
        return $tableStr .= "</tbody>";
    }

    public function setMeta($controlType = 'tableArray', $params = []){
        $this->controlType = $controlType;
        $this->pParams = $params;
    }
    public function getMeta(){
        return mxrAS_HCMakeMeta($this->controlType, '', $this->pParams, '', 0, $this->msg);
    }

    private function getTooltip($cKey,$rowID){
        //cKey is the key in the tooltips
        if(isset($this->tooltips[$cKey])){
            $tt = $this->tooltips[$cKey][0];
            $tooltip = "title='" . $this->tooltips[$cKey][1] . $this->data[$rowID][$tt] . "'";
        } else {
            $tooltip = "";
        }
        return $tooltip;
    }

    public function getMsg(){
        return $this->msg;
    }
    public function getFullTable(){

        //Prepare the data packet to send back to the view.

        //Build the table object...
        $table = [
            'divID'=>$this->divID,
            'tDOM' =>$this->tDOM,
            'data' =>"<table id='{$this->tDOM}' class='display compact'>" . $this->getHeader() . $this->getBody() . $this->getFooter() . "</table>",
            'msg'=>$this->getMsg(),
            'dTableParams'=> $this->dTableParams
        ];
//Output the data.
        return array('meta'=> $this->getMeta(), 'table'=> array($table));
    }

    public function masterSetupTable(){
        //Adjust some of the data...
        //Now we set the meta Params
        

        //Set the cols (we go for all of them);
        $this->setCols([
            'tradedate' => 'Trade Date',
            'rationale' => 'Rationale',
            'editview' => 'View Details'
        ]);
        $this->setRowParams([
            'view_id',
            'dept_id',
            'dept_name'
        ]);

        $this->setTooltips([
            'tradedate' => array('timestamp', "Timestamp : ")
        ]);


    }

}